clear,clc,figure(1),clf
%Physical Parameters
L        =            40000;   %Domain height
k        =                3;   %thermal conductivity
rCp      =        3000*1000;   %Heat capacity times density
rg       =          3000*10;   %Lithostatic gradient
Hr       =         0.5*1e-6;   %Radiogenic heat production
Qs       =          60*1e-3;   %Surface Heat flux
Edot_mm  =                1;   %Erosion rate in mm/yr
y_mark   =  [5000:5000:L,L];   %Initial position of markers
Tsurf    =                0;   %Surface Temperature
%Numerical Parameters
ny       =              200;   %Numerical resolution
CFL      =              0.2;   %Condizione di Courant-Friedrichs-Lewy
nout     =              500;   %Plot every nout steps/store path every nout
%Preprocessing
yr2sec   =        60*60*24*365;   %how many secs in 1 year
Edot     = Edot_mm*1e-3/yr2sec;   %recalculate erosion
dy       =            L/(ny-1);   %Calculate dx
y        =              0:dy:L;   %Domain
dt1      =    dy^2/(k/rCp)*CFL;   %Timestep for diffusion
dt2      =         dy/Edot*CFL;   %Timestep for Erosion
dt       =        min(dt1,dt2);   %Choose the minimum
Edot_m   =        Edot*ones(size(y_mark));
%Initial conditions
T        =  Tsurf+(Qs/k)*y-0.5*Hr/k*y.^2;
T0       =  T;
qflux0   =  -Qs+Hr*y;
%Start time iterations
t     =  0;
i     =  0;
itp   =  0;
Lstop =  0;
while Lstop==0
    i          = i+1;
    t          = t+dt;
    Told       = T;
    T(2:end-1) = Told(2:end-1) + dt*(k/rCp*diff(diff(Told))/dy/dy ...
                                    +Hr/rCp);
    T(1:end-1) = T(1:end-1)     + dt*Edot*diff(T)/dy;
    T(1)       = Tsurf;
    T(end)     = -(qflux0(end)*dy)/k+T(end-1);
    y_mark     = y_mark -Edot_m*dt;
    P_mark     = rg*y_mark;
    T_mark     = interp1(y,T,y_mark,'linear',Tsurf);
    Qsurf      = k*(T(2)-T(1))/dy*1e3;%Value in mW/m2
    Edot_m(y_mark<0)=0;
    if Edot_m(end)==0
        Lstop=1;
    end
    if mod(i,nout)==1
        itp      = itp+1;
        FSS      = 12;
        LWW      = 1.2;
        %Monitor path
        y_path(itp,:) = y_mark;
        P_path(itp,:) = P_mark;
        T_path(itp,:) = T_mark;
        t_path(itp,:) = t*ones(size(P_mark));
        %Visualization
        figure(1),clf
        subplot(121)
        plot(T0,-y/1e3,T,-y/1e3,'LineWidth',LWW),grid on
        title(['t(Myr): ',num2str(t/yr2sec/1e6)])
        xlabel('T (^oC)','FontSize',FSS)
        ylabel('y (km)','FontSize',FSS)
        axis([0 900 -40 0])
        set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
        subplot(122)
        plot(T_path,-y_path/1E3,'LineWidth',LWW),grid on%,hold on
        title(['Qs(mW/m^2): ',num2str(Qsurf)])
        xlabel('T (^oC)','FontSize',FSS)
        ylabel('y (km)','FontSize',FSS)
        axis([0 900 -40 0])
        set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
        drawnow
    end
end

FSS      = 12;
LWW      = 1.2;

%Visualization
subplot(121)
plot(T0,-y/1e3,T,-y/1e3,'LineWidth',LWW),grid on
title(['t(Myr): ',num2str(t/yr2sec/1e6)])
xlabel('T (^oC)','FontSize',FSS)
ylabel('y (km)','FontSize',FSS)
axis([0 900 -40 0])
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
subplot(122)
plot(T_path,-y_path/1E3,'LineWidth',LWW),grid on
title(['Qs(mW/m^2): ',num2str(Qsurf)])
xlabel('T (^oC)','FontSize',FSS)
ylabel('y (km)','FontSize',FSS)
axis([0 900 -40 0])
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)

%Postprocess cooling rates
CR = -diff(T_path,1,1)./diff(t_path,1,1);
CR = [CR]*yr2sec*1e6; %just to keep the same size in last point
TCR= 0.5*(T_path(2:end,:)+T_path(1:end-1,:));
Yp = 0.5*(y_path(2:end,:)+y_path(1:end-1,:));
Pp = 0.5*(P_path(2:end,:)+P_path(1:end-1,:));

figure(2),clf
subplot(221)
plot(T_path,-y_path/1E3,'LineWidth',LWW),hold on
plot(T0,-y/1e3,'--','Color',[0 0 0 0.8],'LineWidth',LWW)
plot(T,-y/1e3,'--','Color',[0 0 0 0.4],'LineWidth',LWW),hold off
hold off
grid on
xlabel('$T$ $(^oC)$','FontSize',FSS,'Interpreter','latex')
ylabel('$y$ $(km)$','FontSize',FSS,'Interpreter','latex')
axis([0 800 -40 0])
axis square
title('A')
text(50,-35,['$Q_s^{final}: $ ',num2str(round(Qsurf)), ...
    ' $ (mW/m^2)$'],'Color',[0 0 0 0.8],'Rotation',0,'FontSize',...
    FSS*0.9,'Interpreter','latex')
text(200,-15,['initial'],'Color',[0 0 0 0.8],'Rotation',-48,'FontSize',...
    FSS,'Interpreter','latex')
text(400,-7.5,['final'],'Color',[0 0 0 0.4],'Rotation',-35,'FontSize',...
    FSS,'Interpreter','latex')
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)

subplot(222)
plot(T_path,P_path/1E9,'LineWidth',LWW),grid on,hold on
plot_reactions(LWW,0.85),hold on
axis square
xlabel('$T$ $(^oC)$','FontSize',FSS,'Interpreter','latex')
ylabel('$P$ $(GPa)$','FontSize',FSS,'Interpreter','latex')
axis([0 800 0 1.5])
hold off
title('B')
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)

subplot(223)
plot(Pp/1e9,CR,'LineWidth',LWW),grid on
xlabel('$P$ $(GPa)$','FontSize',FSS,'Interpreter','latex')
ylabel('$CR$ $(^oC/Myr)$','FontSize',FSS,'Interpreter','latex')
axis([0 1.5 -10 50])
axis square
title('C')
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
%--------
subplot(224)
plot(TCR,CR,'LineWidth',LWW),grid on
axis([50 800 -10 50])
xlabel('$T$ $(^oC)$','FontSize',FSS,'Interpreter','latex')
ylabel('$CR$ $(^oC/Myr)$','FontSize',FSS,'Interpreter','latex')
axis square
title('D')
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
%---------------

set(gcf, 'Position',  [[663 230 882 722]])
stringg = ['print -dpng Results_f4_',num2str(round(Edot_mm)),...
    '.png -r300'];eval(stringg);

%% new figure
figure(3),clf

subplot(121)
plot(Pp/1e9,CR,'LineWidth',LWW),grid on
xlabel('$P$ $(GPa)$','FontSize',FSS,'Interpreter','latex')
ylabel('$CR$ $(^oC/Myr)$','FontSize',FSS,'Interpreter','latex')
axis([0 1.5 -10 500])
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
%--------
subplot(122)
plot(TCR,CR,'LineWidth',LWW),grid on
axis([50 800 -10 500])
xlabel('$T$ $(^oC)$','FontSize',FSS,'Interpreter','latex')
ylabel('$CR$ $(^oC/Myr)$','FontSize',FSS,'Interpreter','latex')
set(gca,'FontSize',FSS*0.8,'LineWidth',LWW*0.8)
set(gcf, 'Position',  [672.6000 423.8000 751.2000 471.2000])

stringg = ['print -dpng Results_f6_',num2str(round(Edot_mm)),...
    '.png -r300'];eval(stringg);
